/*
 * Copyright 2019 NXP
 * All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 */
///////////////////////////////////////////////////////////////////////////////////////////////////
//                                      Includes Section
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <stdlib.h>
#include "pde_preprocessing.h"
#include "mathfp.h"
#include "genfsk_interface.h" /* Get RADIO version */
///////////////////////////////////////////////////////////////////////////////////////////////////
//                                   Defines & Macros Section
///////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef FPDIV
#define FPDIV(x,y,q) (((x) * ((int32_t)(uint32_t)((uint32_t)1U<<(q)))) / (y))
#endif

/* Constant used to compensate internal HW delay when combining phase */
#if defined(RADIO_IS_GEN_3P5)
#define PDE_PREPROCESSING_CIRC_SHIFT (32768*128/128) /* 128/128 Q15 */
#else
#define PDE_PREPROCESSING_CIRC_SHIFT (-16384) /* -64/128 Q15 */
#endif

#define PDE_PREPROCESSING_PHASE_DELTA_HIGH_THRESHOLD              (52429)

#define PDE_PREPROCESSING_PHASE_DELTA_LOW_THRESHOLD               (-52429)

#define PDE_PREPROCESSING_PHASE_CORRECTION_FACTOR                 (2 * 32768)

#define PDE_PREPROCESSING_PHASE_NORMALIZED_HIGH_THRESHOLD         (1 * 32767)

#define PDE_PREPROCESSING_PHASE_NORMALIZED_LOW_THRESHOLD          (-1 * 32768)

#define PDE_PREPROCESSING_IQ_SAT_UPPER_THRESHOLD                  (2000)

#define PDE_PREPROCESSING_IQ_SIMUL_LO_THRESHOLD                   (32)

#define PDE_PREPROCESSING_PHASE_OUTLIER_HIGH_THRESHOLD            (13107)

#ifndef PDE_PREPROCESSING_VALID_FREQS_THRESHOLD
#define PDE_PREPROCESSING_VALID_FREQS_THRESHOLD                   (80U)
#endif

#ifndef PDE_PREPROCESSING_REMOVE_DC
#define PDE_PREPROCESSING_REMOVE_DC (1)
#endif /* PDE_PREPROCESSING_REMOVE_DC */
///////////////////////////////////////////////////////////////////////////////////////////////////
//                                       Typedef Section
///////////////////////////////////////////////////////////////////////////////////////////////////

typedef struct _pde_preprocessing_tag
{
    bool isFrequencyMaskEnable;
    bool isPhaseMaskEnable;
}pre_preproc_enable_t;

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                  Function Prototypes Section
///////////////////////////////////////////////////////////////////////////////////////////////////


///////////////////////////////////////////////////////////////////////////////////////////////////
//                                   Global Constants Section
///////////////////////////////////////////////////////////////////////////////////////////////////


///////////////////////////////////////////////////////////////////////////////////////////////////
//                                   Static Constants Section
///////////////////////////////////////////////////////////////////////////////////////////////////


///////////////////////////////////////////////////////////////////////////////////////////////////
//                                   Global Variables Section
///////////////////////////////////////////////////////////////////////////////////////////////////


///////////////////////////////////////////////////////////////////////////////////////////////////
//                                   Static Variables Section
///////////////////////////////////////////////////////////////////////////////////////////////////

static pre_preproc_enable_t PreprocessingStatus;

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                      Functions Section
///////////////////////////////////////////////////////////////////////////////////////////////////

void PDE_PreprocessingInit(void)
{
    PDE_PreprocessingEnableFrequencyMask(true);

    PDE_PreprocessingEnablePhaseMask(true);
}

#if PDE_PREPROCESSING_REMOVE_DC  == 1
/* compute the averaged I/Q of the IQ samples */
static void PDE_ComputeIqSampleMean(int16_t  *IqSamplesBuffer, 
                                    uint16_t  SamplesPerFreq, 
                                    uint16_t  NumberFrequencies, 
                                    uint32_t *pMaskFreq,
                                    int32_t  *pIMean, 
                                    int32_t  *pQMean) 
{

    int32_t IMean = 0;
    int32_t QMean = 0;
    int32_t IFreqMean = 0;
    int32_t QFreqMean = 0;
    int32_t ISample;
    int32_t QSample;
    int32_t IAbsSample;
    int32_t QAbsSample;
    uint16_t BufferOffset = 0;
    uint16_t FreqOffset = 0;
    uint16_t FreqCounter = 0;
    uint16_t TotalMeanSamples = 0;
    uint16_t MaskByteOffset;
    uint16_t MaskBitOffset;
    
    /* loop through all frequencies */
    while(FreqCounter < NumberFrequencies)
    {
        FreqOffset = 0;
        IFreqMean = 0;
        QFreqMean = 0;

        /* loop through all samples on one frequency */
        while(FreqOffset < SamplesPerFreq)
        {
            ISample = IqSamplesBuffer[BufferOffset];
            QSample = IqSamplesBuffer[BufferOffset + 1U];

            IAbsSample = abs(ISample);
            QAbsSample = abs(QSample);

            /* first check that IQ are not saturated or amplitude is not within bounds */
            if((PreprocessingStatus.isFrequencyMaskEnable == true) && \
                ((IAbsSample > PDE_PREPROCESSING_IQ_SAT_UPPER_THRESHOLD) || (QAbsSample > PDE_PREPROCESSING_IQ_SAT_UPPER_THRESHOLD)))
            {
                IFreqMean = 0;
                QFreqMean = 0;
                break;
            }
            #if 1
            else if((PreprocessingStatus.isFrequencyMaskEnable == true) && \
                ((IAbsSample < PDE_PREPROCESSING_IQ_SIMUL_LO_THRESHOLD) && (QAbsSample < PDE_PREPROCESSING_IQ_SIMUL_LO_THRESHOLD)))
            {
                /* invalid sample */
                IFreqMean = 0;
                QFreqMean = 0;
                break;
            }
            #endif
            else
            {
                /* valid sample */
                IFreqMean += ISample;
                QFreqMean += QSample;
            }

            FreqOffset++;
            BufferOffset += 2U;
        }

        /* compute mask position */
        MaskByteOffset = FreqCounter / 32U;
        MaskBitOffset = FreqCounter % 32U;

        /* if there's a valid freq mean, account for it */
        /* if not, means we need to discard such freq   */
        if(IFreqMean!=0 && QFreqMean!=0)
        {
            /* keep */
            IMean += IFreqMean;
            QMean += QFreqMean;
            TotalMeanSamples++;
            pMaskFreq[MaskByteOffset] |= ((uint32_t)1U << MaskBitOffset);
        }
        else
        {
            /* discard */
            BufferOffset += ((SamplesPerFreq - FreqOffset) * 2U);

            pMaskFreq[MaskByteOffset] &= ~((uint32_t)1U << MaskBitOffset);
        }

        FreqCounter++;
    }

    TotalMeanSamples *= SamplesPerFreq;

    /* return the averaged I/Q */
    *pIMean = IMean / (int32_t)TotalMeanSamples;
    *pQMean = QMean / (int32_t)TotalMeanSamples;
}
#endif

void PDE_PreprocessingSamples(int16_t * IqSamplesBuffer, uint16_t SamplesPerFreq, \
    uint16_t NumberFrequencies, int16_t * PhaseOutput, uint32_t *pMaskFreq)
{
    int64_t PhaseAccumulator = 0;
    int64_t Phase;
    #if PDE_PREPROCESSING_REMOVE_DC == 1
    int32_t IMean = 0;
    int32_t QMean = 0;
    #endif /* PDE_PREPROCESSING_REMOVE_DC */
    int16_t IDcRemoved = 0;
    int16_t QDcRemoved = 0;
    int32_t SamplesPerFreqFp;
    int32_t CorrectedPhase;
    int32_t PreviousPhaseCalc;
    int32_t PhaseDelta;
    int32_t PhaseNormalized;
    int32_t AveragedPhase;
    int32_t PhaseMax = INT32_MIN;
    int32_t PhaseMin = INT32_MAX;
    uint16_t BufferOffset = 0;
    uint16_t FreqOffset = 0;
    uint16_t OutputPhaseOffset = 0;
    uint16_t MaskByteOffset;
    uint16_t MaskBitOffset;

    /* assume IQ is interleaved and 16-bit sign extended */

    /* First step: calculate mean for each I and Q MD and RD    */
    /* mean will be calcualted only from valid IQ samples       */

    #if PDE_PREPROCESSING_REMOVE_DC  == 1
    PDE_ComputeIqSampleMean(IqSamplesBuffer, 
                            SamplesPerFreq, 
                            NumberFrequencies, 
                            pMaskFreq,
                           &IMean, 
                           &QMean);
    #else

    /* if DC Removal is disabled, then mark all the freqs as available */
    FreqOffset = 0;

    while(FreqOffset < NumberFrequencies)
    {
        MaskByteOffset = FreqOffset / 32U;
        MaskBitOffset = FreqOffset % 32U;

        pMaskFreq[MaskByteOffset] |= ((uint32_t)1U << MaskBitOffset);

        FreqOffset++;
    }
    #endif /* PDE_PREPROCESSING_REMOVE_DC */

    /* Second step: subtract mean from each IQ (removes DC), compute phase and remove 2 pi wraps    */
    /* Third step: average phase for every frequency to generate a single data point per frequency  */

    /* Reset buffer offset */
    BufferOffset = 0;

    SamplesPerFreqFp = (int32_t)(uint32_t)((uint32_t)SamplesPerFreq << 15U);

    while(OutputPhaseOffset < NumberFrequencies)
    {
        /* Just calculate phase when IQ are within threshold */
        MaskByteOffset = OutputPhaseOffset / 32U;
        MaskBitOffset = OutputPhaseOffset % 32U;

        if( (pMaskFreq[MaskByteOffset] & ((uint32_t)1U << MaskBitOffset)) != 0U)
        {
            FreqOffset = 0;

            while(FreqOffset < SamplesPerFreq)
            {
                #if PDE_PREPROCESSING_REMOVE_DC == 1
                IDcRemoved = (int16_t)(IqSamplesBuffer[BufferOffset] - IMean);
                QDcRemoved = (int16_t)(IqSamplesBuffer[BufferOffset + 1U] - QMean);
                #else
                IDcRemoved = IqSamplesBuffer[BufferOffset];
                QDcRemoved = IqSamplesBuffer[BufferOffset + 1U];
                #endif /* PDE_PREPROCESSING_REMOVE_DC */

                Phase = atan2fp(QDcRemoved,IDcRemoved);
                /* output is required in Q15, atans is Q12 */
                Phase *= 8;
                /* Normalize phase to Pi in Q15 */
                PhaseNormalized =  (int32_t)FPDIV(Phase,0x1921F,15U);

                /* Skip the first phase of each freq */
                if(FreqOffset != 0U)
                {
                    PhaseDelta = PhaseNormalized - PreviousPhaseCalc;

                    if(PhaseDelta > PDE_PREPROCESSING_PHASE_DELTA_HIGH_THRESHOLD)
                    {
                      CorrectedPhase = PhaseNormalized - PDE_PREPROCESSING_PHASE_CORRECTION_FACTOR;
                    }
                    else if(PhaseDelta < PDE_PREPROCESSING_PHASE_DELTA_LOW_THRESHOLD)
                    {
                      CorrectedPhase = PhaseNormalized + PDE_PREPROCESSING_PHASE_CORRECTION_FACTOR;
                    }
                    else
                    {
                      CorrectedPhase = PhaseNormalized;
                    }

                    PreviousPhaseCalc = CorrectedPhase;
                }
                else
                {
                    PreviousPhaseCalc = PhaseNormalized;
                    CorrectedPhase = PhaseNormalized;
                }

                /* average phase samples at each frequency */
                PhaseAccumulator += (int64_t)CorrectedPhase;

                /* Keep track if MIN and MAX for outlier detection */
                if(CorrectedPhase < PhaseMin)
                {
                    PhaseMin = CorrectedPhase;
                }

                if(CorrectedPhase > PhaseMax)
                {
                    PhaseMax = CorrectedPhase;
                }

                FreqOffset++;
                BufferOffset += 2U;
            }

            /* we're done with phase calculation and accumulate. Now look for outliers  */
            /* verify if the frequency max/min phase are within 0.4 rads appart         */
            /* if they are, means is an outlier, mark the frequency as not valid        */
            /* move to the next one                                                     */
            PhaseDelta = PhaseMax - PhaseMin;

            if((PreprocessingStatus.isPhaseMaskEnable == false) || ((PreprocessingStatus.isPhaseMaskEnable == true)\
                && (PhaseDelta < PDE_PREPROCESSING_PHASE_OUTLIER_HIGH_THRESHOLD)))
            {
                AveragedPhase = (int32_t)FPDIV(PhaseAccumulator,SamplesPerFreqFp,15U);

                if(AveragedPhase > PDE_PREPROCESSING_PHASE_NORMALIZED_HIGH_THRESHOLD)
                {
                    AveragedPhase = AveragedPhase - PDE_PREPROCESSING_PHASE_CORRECTION_FACTOR;
                }
                else if(AveragedPhase < PDE_PREPROCESSING_PHASE_NORMALIZED_LOW_THRESHOLD)
                {
                    AveragedPhase = AveragedPhase + PDE_PREPROCESSING_PHASE_CORRECTION_FACTOR;
                }
                else
                {
                    /* MISRA rule 15.7 */
                }
            }
            else
            {
                AveragedPhase = 0;

                pMaskFreq[MaskByteOffset] &= ~((uint32_t)1U << MaskBitOffset);
            }

            PhaseAccumulator = 0;
            PhaseMax = INT32_MIN;
            PhaseMin = INT32_MAX;
        }
        else
        {
            AveragedPhase = 0;

            /* BufferOffset moves in 2 as is I and Q */
            BufferOffset += (SamplesPerFreq * 2U);
            PhaseAccumulator = 0;
        }

        PhaseOutput[OutputPhaseOffset] = (int16_t)AveragedPhase;

        OutputPhaseOffset++;
    }
}

/*
 * Combine RD and MD phases to keep only the portion that varies over time/distance.
 * Additionaly, apply an offset to compensate time spent in various hardware blocks.
 * Note that the offset depends on IQ capture point. It is determined by characterization, not by design.
 */
void PDE_PreprocessingCombinePhases(int16_t * MdPhases, int16_t * RdPhases, int16_t * PhaseOutput,
                                    uint16_t NumberOfFrequencies)
{
    int32_t SumPhase;
    uint32_t PhaseIndex = 0;
    /* Initial index value in matlab preprocessing is 2 */
    int32_t shiftOffset = 2 * PDE_PREPROCESSING_CIRC_SHIFT;

    while(PhaseIndex < NumberOfFrequencies)
    {
        /* shiftOffset wrapped in [-1,1[ */
        if (shiftOffset > PDE_PREPROCESSING_PHASE_NORMALIZED_HIGH_THRESHOLD)
        {
            shiftOffset -= PDE_PREPROCESSING_PHASE_CORRECTION_FACTOR;
        }
        else if (shiftOffset < PDE_PREPROCESSING_PHASE_NORMALIZED_HIGH_THRESHOLD)
        {
            shiftOffset += PDE_PREPROCESSING_PHASE_CORRECTION_FACTOR;
        }
        else
        {
            /* MISRA rule 15.7 */
        }

        /* only add phase when both vectors have values, otherwise, means that phase is bad */
        if((MdPhases[PhaseIndex] != 0) && (RdPhases[PhaseIndex] != 0))
        {
            SumPhase = (int32_t)MdPhases[PhaseIndex] + (int32_t)RdPhases[PhaseIndex];

            /* Phi shift */
            SumPhase += shiftOffset;

            /* Phi normalization wrapped in [-1,1[ */
            if (SumPhase > PDE_PREPROCESSING_PHASE_NORMALIZED_HIGH_THRESHOLD)
            {
                /* cap to the high limit */
                SumPhase = SumPhase - PDE_PREPROCESSING_PHASE_CORRECTION_FACTOR;
            }
            else if (SumPhase < PDE_PREPROCESSING_PHASE_NORMALIZED_LOW_THRESHOLD)
            {
                /* cap to the low limit */
                SumPhase = SumPhase + PDE_PREPROCESSING_PHASE_CORRECTION_FACTOR;
            }
            else
            {
                /* MISRA rule 15.7 */
            }

            PhaseOutput[PhaseIndex] = (int16_t)SumPhase;
        }
        else
        {
            PhaseOutput[PhaseIndex] = 0;
        }

        /* shiftOffset = PDE_PREPROCESSING_CIRC_SHIFT*(2+phaseIndex) */
        shiftOffset += PDE_PREPROCESSING_CIRC_SHIFT;

        PhaseIndex++;
    }
}

bool PDE_PreprocessingCombineFreqMasks(uint32_t * pLocalFreqMask,uint32_t * pRemoteFreqMask,\
    uint32_t * pFinalFreqMask, uint16_t NumberFrequencies)
{
    bool isValidMeasurement = true;
    uint16_t MaskSlots;
    uint16_t MaskSlotsCounter = 0;
    uint16_t FreqValidCounter = 0;
    uint16_t FreqMaskOffset = 0;

    MaskSlots = (NumberFrequencies / 32U);

    if( (NumberFrequencies % 32U) != 0U )
    {
        MaskSlots += 1U;
    }

    /* AND both masks to nuke frequencies which are bad at either side */

    while(MaskSlotsCounter < MaskSlots)
    {
        pFinalFreqMask[MaskSlotsCounter] = pLocalFreqMask[MaskSlotsCounter] & pRemoteFreqMask[MaskSlotsCounter];

        FreqMaskOffset = 0;

        while(FreqMaskOffset < 32U)
        {
            if( (pFinalFreqMask[MaskSlotsCounter] & ((uint32_t)1U << FreqMaskOffset)) != 0U)
            {
                /* frequency ok */
                FreqValidCounter++;
            }

            FreqMaskOffset++;
        }

        MaskSlotsCounter++;
    }

    /* PDE_VALID_FREQS_THRESHOLD set by a various measurement analysis */
    if(FreqValidCounter < PDE_PREPROCESSING_VALID_FREQS_THRESHOLD)
    {
        /* invalidate the measurement as too few valid freq. */
        isValidMeasurement = false;
    }

    return (isValidMeasurement);
}

void PDE_PreprocessingEnableFrequencyMask(bool isFrequencyMaskEnable)
{
    /* set frequency mask enabling */
    PreprocessingStatus.isFrequencyMaskEnable = isFrequencyMaskEnable;
}

void PDE_PreprocessingEnablePhaseMask(bool isPhaseMaskEnable)
{
    /* get phase setting */
    PreprocessingStatus.isPhaseMaskEnable = isPhaseMaskEnable;
}

bool PDE_PreprocessingGetFrequencyMaskStatus(void)
{
    /* get frequency mask setting */
    return(PreprocessingStatus.isFrequencyMaskEnable);
}

bool PDE_PreprocessingGetPhaseMaskStatus(void)
{
    /* get phase mask setting */
    return(PreprocessingStatus.isPhaseMaskEnable);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
// EOF
///////////////////////////////////////////////////////////////////////////////////////////////////
